<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\Currency;
use App\Models\ExchangeRate;
use App\Models\Remittance;
use App\Models\Customer;
use App\Models\User;
use Carbon\Carbon;

class ReportTest extends TestCase
{
    use RefreshDatabase;

    protected $fromCurrency;
    protected $toCurrency;
    protected $customer;
    protected $user;
    protected $remittance;
    protected $exchangeRate;

    /**
     * الإعداد قبل كل اختبار
     */
    protected function setUp(): void
    {
        parent::setUp();
        
        // إنشاء عملات للاختبار
        $this->fromCurrency = Currency::create([
            'name' => 'الدولار الأمريكي',
            'code' => 'USD',
            'symbol' => '$',
            'country' => 'الولايات المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 1
        ]);

        $this->toCurrency = Currency::create([
            'name' => 'اليورو',
            'code' => 'EUR',
            'symbol' => '€',
            'country' => 'الاتحاد الأوروبي',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 2
        ]);
        
        // إنشاء عميل للاختبار
        $this->customer = Customer::create([
            'name' => 'أحمد محمد',
            'phone' => '0123456789',
            'email' => 'ahmed@example.com',
            'address' => 'شارع الرياض، الرياض',
            'id_number' => 'ID12345678',
            'id_type' => 'بطاقة هوية',
            'nationality' => 'سعودي',
            'notes' => 'عميل اختبار'
        ]);
        
        // إنشاء مستخدم للاختبار
        $this->user = User::factory()->create();
        
        // إنشاء سعر صرف للاختبار
        $this->exchangeRate = ExchangeRate::create([
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'buy_rate' => 0.85,
            'sell_rate' => 0.87,
            'is_active' => true,
            'effective_date' => now(),
            'updated_by' => $this->user->id
        ]);
        
        // إنشاء حوالة للاختبار
        $this->remittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);
    }

    /**
     * اختبار عرض صفحة التقارير الرئيسية
     */
    public function test_index_displays_reports_page()
    {
        // زيارة صفحة التقارير الرئيسية
        $response = $this->get(route('reports.index'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود روابط التقارير المختلفة في الصفحة
        $response->assertSee('تقرير الحوالات المالية');
        $response->assertSee('تقرير أسعار الصرف');
        $response->assertSee('تقرير الإيرادات');
        $response->assertSee('تقرير العملاء');
        $response->assertSee('تقرير ملخص الأداء');
    }

    /**
     * اختبار عرض تقرير الحوالات المالية
     */
    public function test_remittances_report_displays_remittances_data()
    {
        // زيارة صفحة تقرير الحوالات المالية
        $response = $this->get(route('reports.remittances'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات الحوالات في التقرير
        $response->assertSee($this->remittance->reference_number);
        $response->assertSee('محمد علي');
        $response->assertSee('مصر');
        $response->assertSee('1000');
    }

    /**
     * اختبار عرض تقرير أسعار الصرف
     */
    public function test_exchange_rates_report_displays_exchange_rates_data()
    {
        // زيارة صفحة تقرير أسعار الصرف
        $response = $this->get(route('reports.exchange-rates'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات أسعار الصرف في التقرير
        $response->assertSee('USD');
        $response->assertSee('EUR');
        $response->assertSee('0.85');
        $response->assertSee('0.87');
    }

    /**
     * اختبار عرض تقرير الإيرادات
     */
    public function test_revenue_report_displays_revenue_data()
    {
        // زيارة صفحة تقرير الإيرادات
        $response = $this->get(route('reports.revenue'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات الإيرادات في التقرير
        $response->assertSee('الإيرادات');
        $response->assertSee('رسوم الحوالات');
    }

    /**
     * اختبار عرض تقرير العملاء
     */
    public function test_customers_report_displays_customers_data()
    {
        // زيارة صفحة تقرير العملاء
        $response = $this->get(route('reports.customers'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات العملاء في التقرير
        $response->assertSee('أحمد محمد');
    }

    /**
     * اختبار عرض تقرير ملخص الأداء
     */
    public function test_performance_summary_displays_performance_data()
    {
        // زيارة صفحة تقرير ملخص الأداء
        $response = $this->get(route('reports.performance'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات الأداء في التقرير
        $response->assertSee('ملخص الأداء');
        $response->assertSee('إحصائيات اليوم');
        $response->assertSee('إحصائيات الأسبوع');
        $response->assertSee('إحصائيات الشهر');
    }

    /**
     * اختبار تصدير تقرير الحوالات
     */
    public function test_export_remittances_report_generates_csv_file()
    {
        // زيارة صفحة تصدير تقرير الحوالات
        $response = $this->get(route('reports.remittances.export'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من نوع المحتوى
        $response->assertHeader('Content-Type', 'text/csv; charset=UTF-8');

        // التحقق من وجود بيانات الحوالات في ملف CSV
        $content = $response->getContent();
        $this->assertStringContainsString('رقم المرجع', $content);
        $this->assertStringContainsString('تاريخ الإنشاء', $content);
        $this->assertStringContainsString('المرسل', $content);
        $this->assertStringContainsString('المستلم', $content);
        $this->assertStringContainsString($this->remittance->reference_number, $content);
        $this->assertStringContainsString('محمد علي', $content);
        $this->assertStringContainsString('مصر', $content);
        $this->assertStringContainsString('USD', $content);
        $this->assertStringContainsString('EUR', $content);
    }

    /**
     * اختبار تصفية تقرير الحوالات حسب الحالة
     */
    public function test_remittances_report_can_be_filtered_by_status()
    {
        // إنشاء حوالة مكتملة للاختبار
        $completedRemittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'سعيد خالد',
            'receiver_phone' => '0567891234',
            'receiver_address' => 'شارع دبي، دبي',
            'receiver_id_number' => 'ID45678901',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 2000,
            'fee' => 30,
            'exchange_rate' => 0.85,
            'total_amount' => 2030,
            'status' => Remittance::STATUS_COMPLETED,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'الإمارات',
            'destination_city' => 'دبي',
            'transfer_purpose' => 'استثمار',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(1),
            'actual_delivery_date' => Carbon::now(),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // زيارة صفحة تقرير الحوالات مع تصفية حسب الحالة المكتملة
        $response = $this->get(route('reports.remittances', ['status' => Remittance::STATUS_COMPLETED]));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود الحوالة المكتملة وعدم وجود الحوالة المعلقة
        $response->assertSee($completedRemittance->reference_number);
        $response->assertSee('سعيد خالد');
        $response->assertSee('الإمارات');
        $response->assertSee('2000');
        $response->assertDontSee($this->remittance->reference_number);
    }
}
