<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\Remittance;
use App\Models\User;
use Carbon\Carbon;

class RemittanceTest extends TestCase
{
    use RefreshDatabase;

    protected $fromCurrency;
    protected $toCurrency;
    protected $customer;
    protected $user;

    /**
     * الإعداد قبل كل اختبار
     */
    protected function setUp(): void
    {
        parent::setUp();
        
        // إنشاء عملات للاختبار
        $this->fromCurrency = Currency::create([
            'name' => 'الدولار الأمريكي',
            'code' => 'USD',
            'symbol' => '$',
            'country' => 'الولايات المتحدة',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 1
        ]);

        $this->toCurrency = Currency::create([
            'name' => 'اليورو',
            'code' => 'EUR',
            'symbol' => '€',
            'country' => 'الاتحاد الأوروبي',
            'is_active' => true,
            'decimal_places' => 2,
            'sort_order' => 2
        ]);
        
        // إنشاء عميل للاختبار
        $this->customer = Customer::create([
            'name' => 'أحمد محمد',
            'phone' => '0123456789',
            'email' => 'ahmed@example.com',
            'address' => 'شارع الرياض، الرياض',
            'id_number' => 'ID12345678',
            'id_type' => 'بطاقة هوية',
            'nationality' => 'سعودي',
            'notes' => 'عميل اختبار'
        ]);
        
        // إنشاء مستخدم للاختبار
        $this->user = User::factory()->create();
    }

    /**
     * اختبار عرض قائمة الحوالات
     */
    public function test_index_displays_remittances_list()
    {
        // إنشاء حوالة للاختبار
        $remittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // زيارة صفحة قائمة الحوالات
        $response = $this->get(route('remittances.index'));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود الحوالة في الصفحة
        $response->assertSee($remittance->reference_number);
        $response->assertSee('محمد علي');
        $response->assertSee('مصر');
        $response->assertSee('معلقة');
    }

    /**
     * اختبار إنشاء حوالة جديدة
     */
    public function test_store_creates_new_remittance()
    {
        // بيانات الحوالة الجديدة
        $remittanceData = [
            'sender_id' => $this->customer->id,
            'receiver_name' => 'خالد سعيد',
            'receiver_phone' => '0567891234',
            'receiver_address' => 'شارع عمان، عمان',
            'receiver_id_number' => 'ID45678901',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 500,
            'fee' => 10,
            'exchange_rate' => 0.85,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'الأردن',
            'destination_city' => 'عمان',
            'transfer_purpose' => 'تعليم',
            'notes' => 'حوالة اختبار',
            'expected_delivery_date' => Carbon::now()->addDays(1)->format('Y-m-d')
        ];

        // إرسال طلب إنشاء حوالة جديدة
        $response = $this->actingAs($this->user)
            ->post(route('remittances.store'), $remittanceData);

        // التحقق من إعادة التوجيه بعد الإنشاء
        $response->assertRedirect(route('remittances.show', 1));

        // التحقق من وجود الحوالة في قاعدة البيانات
        $this->assertDatabaseHas('remittances', [
            'sender_id' => $this->customer->id,
            'receiver_name' => 'خالد سعيد',
            'destination_country' => 'الأردن',
            'amount' => 500,
            'fee' => 10,
            'status' => Remittance::STATUS_PENDING
        ]);
    }

    /**
     * اختبار عرض تفاصيل حوالة
     */
    public function test_show_displays_remittance_details()
    {
        // إنشاء حوالة للاختبار
        $remittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // زيارة صفحة تفاصيل الحوالة
        $response = $this->get(route('remittances.show', $remittance->id));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود تفاصيل الحوالة في الصفحة
        $response->assertSee($remittance->reference_number);
        $response->assertSee('محمد علي');
        $response->assertSee('0987654321');
        $response->assertSee('مصر');
        $response->assertSee('القاهرة');
        $response->assertSee('1000');
        $response->assertSee('0.85');
        $response->assertSee('دعم عائلي');
    }

    /**
     * اختبار تحديث حوالة
     */
    public function test_update_modifies_remittance()
    {
        // إنشاء حوالة للاختبار
        $remittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // بيانات التحديث
        $updatedData = [
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع الإسكندرية، الإسكندرية',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1200,
            'fee' => 25,
            'exchange_rate' => 0.85,
            'status' => Remittance::STATUS_PROCESSING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'الإسكندرية',
            'transfer_purpose' => 'دعم عائلي',
            'notes' => 'تم تحديث العنوان والمبلغ',
            'expected_delivery_date' => Carbon::now()->addDays(2)->format('Y-m-d')
        ];

        // إرسال طلب تحديث الحوالة
        $response = $this->actingAs($this->user)
            ->put(route('remittances.update', $remittance->id), $updatedData);

        // التحقق من إعادة التوجيه بعد التحديث
        $response->assertRedirect(route('remittances.show', $remittance->id));

        // التحقق من تحديث الحوالة في قاعدة البيانات
        $this->assertDatabaseHas('remittances', [
            'id' => $remittance->id,
            'amount' => 1200,
            'fee' => 25,
            'status' => Remittance::STATUS_PROCESSING,
            'destination_city' => 'الإسكندرية',
            'notes' => 'تم تحديث العنوان والمبلغ'
        ]);
    }

    /**
     * اختبار تغيير حالة الحوالة
     */
    public function test_update_status_changes_remittance_status()
    {
        // إنشاء حوالة للاختبار
        $remittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // بيانات تحديث الحالة
        $statusUpdateData = [
            'status' => Remittance::STATUS_COMPLETED,
            'notes' => 'تم تسليم الحوالة للمستلم'
        ];

        // إرسال طلب تحديث حالة الحوالة
        $response = $this->actingAs($this->user)
            ->post(route('remittances.update-status', $remittance->id), $statusUpdateData);

        // التحقق من إعادة التوجيه بعد التحديث
        $response->assertRedirect(route('remittances.show', $remittance->id));

        // التحقق من تحديث حالة الحوالة في قاعدة البيانات
        $this->assertDatabaseHas('remittances', [
            'id' => $remittance->id,
            'status' => Remittance::STATUS_COMPLETED
        ]);

        // التحقق من تحديث تاريخ التسليم الفعلي
        $updatedRemittance = Remittance::find($remittance->id);
        $this->assertNotNull($updatedRemittance->actual_delivery_date);
    }

    /**
     * اختبار طباعة إيصال الحوالة
     */
    public function test_print_receipt_displays_remittance_receipt()
    {
        // إنشاء حوالة للاختبار
        $remittance = Remittance::create([
            'reference_number' => 'RM' . time() . rand(1000, 9999),
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // زيارة صفحة طباعة إيصال الحوالة
        $response = $this->get(route('remittances.print', $remittance->id));

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات الإيصال في الصفحة
        $response->assertSee($remittance->reference_number);
        $response->assertSee('أحمد محمد'); // اسم المرسل
        $response->assertSee('محمد علي'); // اسم المستلم
        $response->assertSee('1000'); // المبلغ
        $response->assertSee('20'); // الرسوم
        $response->assertSee('مصر'); // بلد الوجهة
        $response->assertSee($remittance->transfer_code); // رمز التحويل
    }

    /**
     * اختبار التحقق من حالة الحوالة
     */
    public function test_check_status_finds_remittance()
    {
        // إنشاء حوالة للاختبار
        $remittance = Remittance::create([
            'reference_number' => 'RM12345678',
            'sender_id' => $this->customer->id,
            'receiver_name' => 'محمد علي',
            'receiver_phone' => '0987654321',
            'receiver_address' => 'شارع القاهرة، القاهرة',
            'receiver_id_number' => 'ID98765432',
            'from_currency_id' => $this->fromCurrency->id,
            'to_currency_id' => $this->toCurrency->id,
            'amount' => 1000,
            'fee' => 20,
            'exchange_rate' => 0.85,
            'total_amount' => 1020,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => Remittance::PAYMENT_CASH,
            'destination_country' => 'مصر',
            'destination_city' => 'القاهرة',
            'transfer_purpose' => 'دعم عائلي',
            'transfer_code' => 'ABCD1234',
            'expected_delivery_date' => Carbon::now()->addDays(2),
            'created_by' => $this->user->id,
            'updated_by' => $this->user->id
        ]);

        // إرسال طلب التحقق من حالة الحوالة باستخدام الرقم المرجعي
        $response = $this->post(route('remittances.check-status.post'), [
            'reference' => 'RM12345678'
        ]);

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات الحوالة في الصفحة
        $response->assertSee('RM12345678');
        $response->assertSee('محمد علي');
        $response->assertSee('معلقة');

        // إرسال طلب التحقق من حالة الحوالة باستخدام رمز التحويل
        $response = $this->post(route('remittances.check-status.post'), [
            'reference' => 'ABCD1234'
        ]);

        // التحقق من نجاح الاستجابة
        $response->assertStatus(200);

        // التحقق من وجود بيانات الحوالة في الصفحة
        $response->assertSee('RM12345678');
        $response->assertSee('محمد علي');
        $response->assertSee('معلقة');
    }
}
