@extends('layouts.app')

@section('title', 'إدارة المعاملات')

@section('content')
<div class="container-fluid">
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">قائمة المعاملات</h5>
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addTransactionModal">
                <i class="bi bi-plus-lg"></i> إضافة معاملة جديدة
            </button>
        </div>
        <div class="card-body">
            <!-- فلاتر البحث -->
            <div class="row mb-3">
                <div class="col-md-12">
                    <form action="{{ route('transactions.index') }}" method="GET" class="row g-3">
                        <div class="col-md-2">
                            <input type="text" class="form-control" name="transaction_number" placeholder="رقم المعاملة" value="{{ request('transaction_number') }}">
                        </div>
                        <div class="col-md-2">
                            <select class="form-select" name="from_currency_id">
                                <option value="">العملة المصدر</option>
                                @if(isset($currencies))
                                    @foreach($currencies as $currency)
                                        <option value="{{ $currency->id }}" {{ request('from_currency_id') == $currency->id ? 'selected' : '' }}>{{ $currency->code }}</option>
                                    @endforeach
                                @endif
                            </select>
                        </div>
                        <div class="col-md-2">
                            <select class="form-select" name="to_currency_id">
                                <option value="">العملة الهدف</option>
                                @if(isset($currencies))
                                    @foreach($currencies as $currency)
                                        <option value="{{ $currency->id }}" {{ request('to_currency_id') == $currency->id ? 'selected' : '' }}>{{ $currency->code }}</option>
                                    @endforeach
                                @endif
                            </select>
                        </div>
                        <div class="col-md-2">
                            <select class="form-select" name="status">
                                <option value="">الحالة</option>
                                <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>مكتملة</option>
                                <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>معلقة</option>
                                <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>ملغاة</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <input type="date" class="form-control" name="start_date" placeholder="من تاريخ" value="{{ request('start_date') }}">
                        </div>
                        <div class="col-md-2">
                            <div class="d-flex">
                                <input type="date" class="form-control" name="end_date" placeholder="إلى تاريخ" value="{{ request('end_date') }}">
                                <button type="submit" class="btn btn-primary ms-2">
                                    <i class="bi bi-search"></i>
                                </button>
                                <a href="{{ route('transactions.index') }}" class="btn btn-secondary ms-1">
                                    <i class="bi bi-x-lg"></i>
                                </a>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>رقم المعاملة</th>
                            <th>العميل</th>
                            <th>من</th>
                            <th>إلى</th>
                            <th>سعر الصرف</th>
                            <th>العمولة</th>
                            <th>نوع المعاملة</th>
                            <th>الحالة</th>
                            <th>التاريخ</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        @if(isset($transactions) && count($transactions) > 0)
                            @foreach($transactions as $transaction)
                                <tr>
                                    <td>{{ $transaction->transaction_number }}</td>
                                    <td>{{ $transaction->customer->name ?? 'غير محدد' }}</td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <span class="me-1">{{ $transaction->fromCurrency->symbol }}</span>
                                            {{ number_format($transaction->from_amount, 2) }}
                                            <small class="text-muted ms-1">{{ $transaction->fromCurrency->code }}</small>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <span class="me-1">{{ $transaction->toCurrency->symbol }}</span>
                                            {{ number_format($transaction->to_amount, 2) }}
                                            <small class="text-muted ms-1">{{ $transaction->toCurrency->code }}</small>
                                        </div>
                                    </td>
                                    <td>{{ number_format($transaction->exchange_rate, 4) }}</td>
                                    <td>{{ number_format($transaction->commission, 2) }}</td>
                                    <td>
                                        @if($transaction->transaction_type == 'buy')
                                            <span class="badge bg-info">شراء</span>
                                        @else
                                            <span class="badge bg-primary">بيع</span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($transaction->status == 'completed')
                                            <span class="badge bg-success">مكتملة</span>
                                        @elseif($transaction->status == 'pending')
                                            <span class="badge bg-warning">معلقة</span>
                                        @else
                                            <span class="badge bg-danger">ملغاة</span>
                                        @endif
                                    </td>
                                    <td>{{ $transaction->created_at->format('Y-m-d H:i') }}</td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <button type="button" class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#viewTransactionModal{{ $transaction->id }}">
                                                <i class="bi bi-eye"></i>
                                            </button>
                                            <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editTransactionModal{{ $transaction->id }}">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#deleteTransactionModal{{ $transaction->id }}">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        @else
                            <tr>
                                <td colspan="10" class="text-center">لا توجد معاملات مسجلة</td>
                            </tr>
                        @endif
                    </tbody>
                </table>
            </div>

            <!-- ترقيم الصفحات -->
            @if(isset($transactions) && $transactions instanceof \Illuminate\Pagination\LengthAwarePaginator)
                <div class="d-flex justify-content-center mt-3">
                    {{ $transactions->appends(request()->query())->links() }}
                </div>
            @endif
        </div>
    </div>
</div>

<!-- نافذة إضافة معاملة جديدة -->
<div class="modal fade" id="addTransactionModal" tabindex="-1" aria-labelledby="addTransactionModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form action="{{ route('transactions.store') }}" method="POST">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title" id="addTransactionModalLabel">إضافة معاملة جديدة</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="إغلاق"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="customer_id" class="form-label">العميل</label>
                            <select class="form-select" id="customer_id" name="customer_id">
                                <option value="">-- اختر العميل --</option>
                                @if(isset($customers))
                                    @foreach($customers as $customer)
                                        <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                                    @endforeach
                                @endif
                            </select>
                            <div class="form-text">اختياري: يمكن ترك هذا الحقل فارغًا للمعاملات السريعة</div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="transaction_type" class="form-label">نوع المعاملة <span class="text-danger">*</span></label>
                            <select class="form-select" id="transaction_type" name="transaction_type" required>
                                <option value="">-- اختر نوع المعاملة --</option>
                                <option value="buy">شراء</option>
                                <option value="sell">بيع</option>
                            </select>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="from_currency_id" class="form-label">العملة المصدر <span class="text-danger">*</span></label>
                            <select class="form-select" id="from_currency_id" name="from_currency_id" required>
                                <option value="">-- اختر العملة --</option>
                                @if(isset($currencies))
                                    @foreach($currencies as $currency)
                                        <option value="{{ $currency->id }}">{{ $currency->code }} - {{ $currency->name }}</option>
                                    @endforeach
                                @endif
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="to_currency_id" class="form-label">العملة الهدف <span class="text-danger">*</span></label>
                            <select class="form-select" id="to_currency_id" name="to_currency_id" required>
                                <option value="">-- اختر العملة --</option>
                                @if(isset($currencies))
                                    @foreach($currencies as $currency)
                                        <option value="{{ $currency->id }}">{{ $currency->code }} - {{ $currency->name }}</option>
                                    @endforeach
                                @endif
                            </select>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="from_amount" class="form-label">المبلغ بالعملة المصدر <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="from_amount" name="from_amount" required step="0.01" min="0.01">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="exchange_rate" class="form-label">سعر الصرف <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="exchange_rate" name="exchange_rate" required step="0.000001" min="0.000001">
                            <div class="form-text">سيتم تحديث هذا الحقل تلقائيًا عند اختيار العملات</div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="to_amount" class="form-label">المبلغ بالعملة الهدف</label>
                            <input type="number" class="form-control" id="to_amount" name="to_amount" readonly step="0.01">
                            <div class="form-text">سيتم حساب هذا الحقل تلقائيًا</div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="commission" class="form-label">العمولة</label>
                            <input type="number" class="form-control" id="commission" name="commission" step="0.01" min="0" value="0">
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="payment_method" class="form-label">طريقة الدفع</label>
                            <select class="form-select" id="payment_method" name="payment_method">
                                <option value="cash">نقدي</option>
                                <option value="card">بطاقة</option>
                                <option value="bank_transfer">تحويل بنكي</option>
                                <option value="other">أخرى</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="status" class="form-label">الحالة <span class="text-danger">*</span></label>
                            <select class="form-select" id="status" name="status" required>
                                <option value="completed">مكتملة</option>
                                <option value="pending">معلقة</option>
                            </select>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="notes" class="form-label">ملاحظات</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- نوافذ عرض وتعديل وحذف المعاملات (ستتم إضافتها ديناميكيًا لكل معاملة) -->
@if(isset($transactions))
    @foreach($transactions as $transaction)
        <!-- نافذة عرض تفاصيل المعاملة -->
        <div class="modal fade" id="viewTransactionModal{{ $transaction->id }}" tabindex="-1" aria-labelledby="viewTransactionModalLabel{{ $transaction->id }}" aria-hidden="true">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="viewTransactionModalLabel{{ $transaction->id }}">تفاصيل المعاملة #{{ $transaction->transaction_number }}</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="إغلاق"></button>
                    </div>
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>معلومات المعاملة</h6>
                                <table class="table table-sm table-borderless">
                                    <tr>
                                        <th>رقم المعاملة:</th>
                                        <td>{{ $transaction->transaction_number }}</td>
                                    </tr>
                                    <tr>
                                        <th>نوع المعاملة:</th>
                                        <td>
                                            @if($transaction->transaction_type == 'buy')
                                                <span class="badge bg-info">شراء</span>
                                            @else
                                                <span class="badge bg-primary">بيع</span>
                                            @endif
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>الحالة:</th>
                                        <td>
                                            @if($transaction->status == 'completed')
                                                <span class="badge bg-success">مكتملة</span>
                                            @elseif($transaction->status == 'pending')
                                                <span class="badge bg-warning">معلقة</span>
                                            @else
                                                <span class="badge bg-danger">ملغاة</span>
                                            @endif
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>تاريخ المعاملة:</th>
                                        <td>{{ $transaction->created_at->format('Y-m-d H:i:s') }}</td>
                                    </tr>
                                    <tr>
                                        <th>طريقة الدفع:</th>
                                        <td>{{ $transaction->payment_method }}</td>
                                    </tr>
                                    <tr>
                                        <th>الموظف:</th>
                                        <td>{{ $transaction->user->name ?? 'غير محدد' }}</td>
                                    </tr>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <h6>معلومات العميل</h6>
                                @if($transaction->customer)
                                    <table class="table table-sm table-borderless">
                                        <tr>
                                            <th>الاسم:</th>
                                            <td>{{ $transaction->customer->name }}</td>
                                        </tr>
                                        <tr>
                                            <th>رقم الهاتف:</th>
                                            <td>{{ $transaction->customer->phone ?? 'غير متوفر' }}</td>
                                        </tr>
                                        <tr>
                                            <th>البريد الإلكتروني:</th>
                                            <td>{{ $transaction->customer->email ?? 'غير متوفر' }}</td>
                                        </tr>
                                        <tr>
                                            <th>نوع الهوية:</th>
                                            <td>{{ $transaction->customer->id_type ?? 'غير متوفر' }}</td>
                                        </tr>
                                        <tr>
                                            <th>رقم الهوية:</th>
                                            <td>{{ $transaction->customer->id_number ?? 'غير متوفر' }}</td>
                                        </tr>
                                    </table>
                                @else
                                    <p>معاملة سريعة بدون عميل محدد</p>
                                @endif
                            </div>
                        </div>
                        <hr>
                        <div class="row">
                            <div class="col-md-12">
                                <h6>تفاصيل الصرف</h6>
                                <table class="table table-bordered">
                                    <thead>
                                        <tr>
                                            <th>من العملة</th>
                                            <th>المبلغ</th>
                                            <th>سعر الصرف</th>
                                            <th>إلى العملة</th>
                                            <th>المبلغ</th>
                                            <th>العمولة</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td>{{ $transaction->fromCurrency->code }}</td>
                                            <td>{{ number_format($transaction->from_amount, 2) }}</td>
                                            <td>{{ number_format($transaction->exchange_rate, 6) }}</td>
                                            <td>{{ $transaction->toCurrency->code }}</td>
                                            <td>{{ number_format($transaction->to_amount, 2) }}</td>
                                            <td>{{ number_format($transaction->commission, 2) }}</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        @if($transaction->notes)
                            <div class="row mt-3">
                                <div class="col-md-12">
                                    <h6>ملاحظات</h6>
                                    <p>{{ $transaction->notes }}</p>
                                </div>
                            </div>
                        @endif
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إغلاق</button>
                        <button type="button" class="btn btn-primary" onclick="printTransaction({{ $transaction->id }})">طباعة</button>
                    </div>
                </div>
            </div>
        </div>

        <!-- نافذة تعديل المعاملة -->
        <div class="modal fade" id="editTransactionModal{{ $transaction->id }}" tabindex="-1" aria-labelledby="editTransactionModalLabel{{ $transaction->id }}" aria-hidden="true">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <form action="{{ route('transactions.update', $transaction->id) }}" method="POST">
                        @csrf
                        @method('PUT')
                        <div class="modal-header">
                            <h5 class="modal-title" id="editTransactionModalLabel{{ $transaction->id }}">تعديل المعاملة #{{ $transaction->transaction_number }}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="إغلاق"></button>
                        </div>
                        <div class="modal-body">
                            <!-- حقول التعديل مشابهة لحقول الإضافة مع تعبئة البيانات الحالية -->
                            <div class="alert alert-warning">
                                <i class="bi bi-exclamation-triangle"></i>
                                تحذير: تعديل المعاملة قد يؤثر على السجلات المالية. تأكد من صحة البيانات.
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="edit_status{{ $transaction->id }}" class="form-label">الحالة <span class="text-danger">*</span></label>
                                    <select class="form-select" id="edit_status{{ $transaction->id }}" name="status" required>
                                        <option value="completed" {{ $transaction->status == 'completed' ? 'selected' : '' }}>مكتملة</option>
                                        <option value="pending" {{ $transaction->status == 'pending' ? 'selected' : '' }}>معلقة</option>
                                        <option value="cancelled" {{ $transaction->status == 'cancelled' ? 'selected' : '' }}>ملغاة</option>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="edit_notes{{ $transaction->id }}" class="form-label">ملاحظات</label>
                                    <textarea class="form-control" id="edit_notes{{ $transaction->id }}" name="notes" rows="3">{{ $transaction->notes }}</textarea>
                                </div>
                            </div>
                            <div class="form-text text-center">
                                لتعديل تفاصيل المعاملة الأخرى، يرجى إلغاء هذه المعاملة وإنشاء معاملة جديدة.
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                            <button type="submit" class="btn btn-primary">حفظ التغييرات</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- نافذة حذف المعاملة -->
        <div class="modal fade" id="deleteTransactionModal{{ $transaction->id }}" tabindex="-1" aria-labelledby="deleteTransactionModalLabel{{ $transaction->id }}" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <form action="{{ route('transactions.destroy', $transaction->id) }}" method="POST">
                        @csrf
                        @method('DELETE')
                        <div class="modal-header">
                            <h5 class="modal-title" id="deleteTransactionModalLabel{{ $transaction->id }}">تأكيد الحذف</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="إغلاق"></button>
                        </div>
                        <div class="modal-body">
                            <p>هل أنت متأكد من حذف المعاملة رقم "{{ $transaction->transaction_number }}"؟</p>
                            <div class="alert alert-danger">
                                <i class="bi bi-exclamation-triangle"></i>
                                تحذير: حذف المعاملة سيؤثر على السجلات المالية والتقارير. يفضل تغيير حالة المعاملة إلى "ملغاة" بدلاً من الحذف.
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                            <button type="submit" class="btn btn-danger">تأكيد الحذف</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    @endforeach
@endif
@endsection

@section('scripts')
<script>
    // حساب المبلغ بالعملة الهدف تلقائيًا
    document.addEventListener('DOMContentLoaded', function() {
        const fromAmountInput = document.getElementById('from_amount');
        const exchangeRateInput = document.getElementById('exchange_rate');
        const toAmountInput = document.getElementById('to_amount');
        
        function calculateToAmount() {
            const fromAmount = parseFloat(fromAmountInput.value) || 0;
            const exchangeRate = parseFloat(exchangeRateInput.value) || 0;
            const toAmount = fromAmount * exchangeRate;
            toAmountInput.value = toAmount.toFixed(2);
        }
        
        fromAmountInput.addEventListener('input', calculateToAmount);
        exchangeRateInput.addEventListener('input', calculateToAmount);
        
        // تحديث سعر الصرف عند اختيار العملات
        const fromCurrencySelect = document.getElementById('from_currency_id');
        const toCurrencySelect = document.getElementById('to_currency_id');
        
        function updateExchangeRate() {
            const fromCurrencyId = fromCurrencySelect.value;
            const toCurrencyId = toCurrencySelect.value;
            
            if (fromCurrencyId && toCurrencyId) {
                // هنا يمكن إضافة طلب AJAX للحصول على سعر الصرف الحالي من الخادم
                // لأغراض العرض، نستخدم قيمة افتراضية
                exchangeRateInput.value = '1.0000';
                calculateToAmount();
            }
        }
        
        fromCurrencySelect.addEventListener('change', updateExchangeRate);
        toCurrencySelect.addEventListener('change', updateExchangeRate);
    });
    
    // وظيفة طباعة المعاملة
    function printTransaction(id) {
        const printWindow = window.open('', '_blank');
        const modalContent = document.querySelector(`#viewTransactionModal${id} .modal-body`).innerHTML;
        
        printWindow.document.write(`
            <!DOCTYPE html>
            <html dir="rtl">
            <head>
                <title>طباعة المعاملة</title>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
                <style>
                    body {
                        font-family: 'Tajawal', sans-serif;
                        padding: 20px;
                    }
                    .header {
                        text-align: center;
                        margin-bottom: 20px;
                    }
                    .footer {
                        text-align: center;
                        margin-top: 30px;
                        font-size: 14px;
                    }
                    @media print {
                        @page {
                            size: A4;
                            margin: 10mm;
                        }
                    }
                </style>
            </head>
            <body>
                <div class="header">
                    <h3>CodePlus - نظام الصرافة والحوالات</h3>
                    <h4>إيصال معاملة</h4>
                </div>
                ${modalContent}
                <div class="footer">
                    <p>جميع الحقوق محفوظة © ${new Date().getFullYear()} CodePlus - نظام الصرافة والحوالات</p>
                </div>
                <script>
                    window.onload = function() {
                        window.print();
                        setTimeout(function() {
                            window.close();
                        }, 500);
                    };
                </script>
            </body>
            </html>
        `);
        
        printWindow.document.close();
    }
</script>
@endsection
