<?php

namespace App\Services;

use App\Models\ActivityLog;
use App\Models\Role;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

/**
 * خدمة المصادقة للنظام المصرفي المتكامل
 * تتعامل مع عمليات تسجيل الدخول والتسجيل وإدارة المستخدمين
 */
class AuthService
{

    
    /**
     * المستخدم الحالي
     */
    private $currentUser = null;
    

    /**
     * تحميل بيانات المستخدم الحالي
     * 
     * @param int $userId معرف المستخدم
     * @return bool نتيجة تحميل بيانات المستخدم
     */
    private function loadCurrentUser($userId)
    {
        $user = \App\Models\User::where('id', $userId)->first(); // or User::find($userId);
    
        if ($user) {
            $this->currentUser = $user;
            return true;
        }
    
        return false;
    }
    
    /**
     * تسجيل دخول المستخدم
     * 
     * @param string $username اسم المستخدم
     * @param string $password كلمة المرور
     * @param bool $remember تذكر المستخدم
     * @return bool|array نتيجة تسجيل الدخول
     */
    public function login($username, $password, $remember = false)
    {
        // Attempt to find the user by username or email
        $user = User::where('username', $username)->orWhere('email', $username)->first();
    
        if (!$user) {
            return ['success' => false, 'message' => 'اسم المستخدم أو كلمة المرور غير صحيحة'];
        }
    
        if (!$user->is_active) {
            return ['success' => false, 'message' => 'الحساب غير مفعل'];
        }
    
        // Check the password
        if (!Hash::check($password, $user->password)) {
            $this->logActivity(null, 'login_failed', 'محاولة تسجيل دخول فاشلة: ' . $username);
            return ['success' => false, 'message' => 'اسم المستخدم أو كلمة المرور غير صحيحة'];
        }
    
        // Authenticate the user manually
        Auth::login($user, $remember);
    
        // Update last login time
        $user->last_login = now();
        $user->save();
    
        // Log activity
        $this->logActivity($user->id, 'login', 'تم تسجيل الدخول بنجاح');
    
        return ['success' => true, 'user' => $user];
    }
    
    /**
     * تسجيل خروج المستخدم
     * 
     * @return bool نتيجة تسجيل الخروج
     */
    public function logout()
    {

        if (Auth::check()) {
            $user = Auth::user();

            // Update user_sessions table if you are manually tracking sessions
            DB::table('user_sessions')
                ->where('user_id', $user->id)
                ->whereNull('logout_at')
                ->orderByDesc('login_at')
                ->limit(1)
                ->update(['logout_at' => now()]);

            // Log the logout activity
            $this->logActivity($user->id, 'logout', 'تم تسجيل الخروج بنجاح');

            // Remove remember token from database and cookie if you use custom remember tokens
            if (Cookie::has('remember_token')) {
                list($userId, $token) = explode(':', Cookie::get('remember_token'));

                DB::table('remember_tokens')->where('user_id', $userId)->delete();

                Cookie::queue(Cookie::forget('remember_token'));
            }

            // Laravel's built-in logout
            Auth::logout();

            // Invalidate session
            session()->invalidate();
            session()->regenerateToken(); // Protect against CSRF attacks after logout

            return true;
        }

        return false;
    }
    
    /**
     * تسجيل مستخدم جديد
     * 
     * @param array $userData بيانات المستخدم
     * @return bool|array نتيجة التسجيل
     */
    public function register(array $userData)
    {
        // التحقق من وجود اسم المستخدم أو البريد الإلكتروني
        $exists = User::where('username', $userData['username'])
                    ->orWhere('email', $userData['email'])
                    ->exists();




        if ($exists) {
            return ['success' => false, 'message' => 'اسم المستخدم أو البريد الإلكتروني مستخدم بالفعل'];
        }

        // الحصول على دور المستخدم العادي أو إنشاؤه
        $role = Role::firstOrCreate(
            ['name' => 'مستخدم عادي'],
            ['description' => 'مستخدم عادي للنظام']
        );


        $full_name = $userData['full_name'];
        $username = $userData['username'];
        $email = $userData['email'];
        $password = Hash::make($userData['password']);
        $phone = $userData['phone'];
        $address = $userData['address'];
        $role_id = $role->id;


        // إنشاء المستخدم الجديد
        $user = User::create([
            'full_name' => $full_name,
            'username' => $username,
            'email' => $email,
            'password' => $password,
            'phone' => $phone,
            'address' => $address,
            'role_id' => $role_id,
            'is_active' => true,
        ]);

        if (!$user) {
            return ['success' => false, 'message' => 'حدث خطأ أثناء إنشاء المستخدم'];
        }

        // تسجيل نشاط التسجيل
        $this->logActivity($user->id, 'register', 'تم التسجيل بنجاح');

        return ['success' => true, 'user_id' => $user->id];
    }
    
    /**
     * إنشاء جلسة مستخدم جديدة
     * 
     * @param int $userId معرف المستخدم
     * @return bool نتيجة إنشاء الجلسة
     */
    private function createUserSession($userId)
    {
        $stmt = $this->db->prepare("
            INSERT INTO user_sessions (
                user_id, ip_address, user_agent, login_at
            ) VALUES (
                :user_id, :ip_address, :user_agent, NOW()
            )
        ");
        
        return $stmt->execute([
            'user_id' => $userId,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    }
    
    /**
     * تسجيل نشاط في النظام
     * 
     * @param int|null $userId معرف المستخدم
     * @param string $activityType نوع النشاط
     * @param string $description وصف النشاط
     * @return bool نتيجة تسجيل النشاط
     */
    public function logActivity($userId, $activityType, $description)
    {
        return ActivityLog::create([
            'user_id'      => $userId,
            'activity_type'=> $activityType,
            'description'  => $description,
            'ip_address'   => request()->ip(), // Laravel helper instead of $_SERVER
            'user_agent'   => request()->userAgent(), // Laravel helper instead of $_SERVER
        ]);
    }



    
    /**
     * التحقق من تسجيل دخول المستخدم
     * 
     * @return bool حالة تسجيل الدخول
     */
    public function isLoggedIn()
    {
        return $this->currentUser !== null;
    }
    
    /**
     * الحصول على بيانات المستخدم الحالي
     * 
     * @return array|null بيانات المستخدم
     */
    public function getCurrentUser()
    {
        return $this->currentUser;
    }
    
    /**
     * التحقق من صلاحيات المستخدم
     * 
     * @param string $permission الصلاحية المطلوبة
     * @return bool نتيجة التحقق
     */
    public function hasPermission($permission)
    {
        if (!$this->isLoggedIn()) {
            return false;
        }

        $role = Role::with('permissions')->find($this->currentUser['role_id']);
        
        if (!$role) {
            return false;
        }

        return $role->permissions->contains('name', $permission);
    }
}
