<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    use HasFactory;
    
  /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];
    
    /**
     * الحصول على قيمة إعداد معين
     *
     * @param string $key مفتاح الإعداد
     * @param mixed $default القيمة الافتراضية في حال عدم وجود الإعداد
     * @return mixed
     */
    public static function getValue($key, $default = null)
    {
        // محاولة الحصول على القيمة من الكاش
        return Cache::remember('setting_'.$key, 60*60, function() use ($key, $default) {
            $setting = self::where('key', $key)->first();
            
            if (!$setting) {
                return $default;
            }
            
            // تحويل القيمة حسب نوعها
            switch ($setting->type) {
                case 'integer':
                    return (int) $setting->value;
                case 'float':
                    return (float) $setting->value;
                case 'boolean':
                    return (bool) $setting->value;
                case 'array':
                    return json_decode($setting->value, true);
                case 'object':
                    return json_decode($setting->value);
                default:
                    return $setting->value;
            }
        });
    }
    
    /**
     * تعيين قيمة إعداد معين
     *
     * @param string $key مفتاح الإعداد
     * @param mixed $value القيمة الجديدة
     * @return bool
     */
    public static function setValue($key, $value)
    {
        $setting = self::where('key', $key)->first();
        
        if (!$setting) {
            return false;
        }
        
        // تحويل القيمة حسب نوعها
        if ($setting->type == 'array' || $setting->type == 'object') {
            $value = json_encode($value);
        }
        
        $setting->value = $value;
        $result = $setting->save();
        
        // حذف الكاش
        Cache::forget('setting_'.$key);
        
        return $result;
    }
    
    /**
     * الحصول على جميع الإعدادات حسب المجموعة
     *
     * @param string $group مجموعة الإعدادات
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getByGroup($group)
    {
        return self::where('group', $group)->get();
    }
    
    /**
     * الحصول على جميع الإعدادات العامة
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getPublicSettings()
    {
        return self::where('is_public', true)->get();
    }
}
