<?php

namespace App\Http\Controllers;

use App\Models\Remittance;
use App\Models\Customer;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class RemittanceController extends Controller
{
    /**
     * عرض قائمة الحوالات المالية
     */
    public function index(Request $request)
    {
        $status = $request->input('status');
        $query = Remittance::with(['sender', 'fromCurrency', 'toCurrency', 'createdBy']);
        
        // تصفية حسب الحالة إذا تم تحديدها
        if ($status) {
            $query->withStatus($status);
        }
        
        $remittances = $query->orderBy('created_at', 'desc')->paginate(15);
        
        // الحصول على إحصائيات الحوالات
        $stats = [
            'total' => Remittance::count(),
            'pending' => Remittance::pending()->count(),
            'processing' => Remittance::processing()->count(),
            'completed' => Remittance::completed()->count(),
            'cancelled' => Remittance::cancelled()->count(),
            'refunded' => Remittance::refunded()->count(),
            'on_hold' => Remittance::onHold()->count(),
        ];
        
        return view('remittances.index', compact('remittances', 'stats', 'status'));
    }

    /**
     * عرض نموذج إنشاء حوالة جديدة
     */
    public function create()
    {
        $customers = Customer::orderBy('name')->get();
        $currencies = Currency::active()->ordered()->get();
        $paymentMethods = [
            Remittance::PAYMENT_CASH => 'نقداً',
            Remittance::PAYMENT_BANK_TRANSFER => 'تحويل بنكي',
            Remittance::PAYMENT_CREDIT_CARD => 'بطاقة ائتمان',
            Remittance::PAYMENT_MOBILE_WALLET => 'محفظة إلكترونية'
        ];
        
        return view('remittances.create', compact('customers', 'currencies', 'paymentMethods'));
    }

    /**
     * حفظ حوالة جديدة في قاعدة البيانات
     */
    public function store(Request $request)
    {
        // التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'sender_id' => 'required|exists:customers,id',
            'receiver_name' => 'required|string|max:255',
            'receiver_phone' => 'required|string|max:20',
            'receiver_address' => 'required|string|max:500',
            'receiver_id_number' => 'required|string|max:50',
            'from_currency_id' => 'required|exists:currencies,id',
            'to_currency_id' => 'required|exists:currencies,id|different:from_currency_id',
            'amount' => 'required|numeric|min:0.01',
            'fee' => 'required|numeric|min:0',
            'exchange_rate' => 'required|numeric|min:0.000001',
            'payment_method' => 'required|string',
            'destination_country' => 'required|string|max:100',
            'destination_city' => 'required|string|max:100',
            'transfer_purpose' => 'required|string|max:255',
            'notes' => 'nullable|string|max:1000',
            'expected_delivery_date' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return redirect()->route('remittances.create')
                ->withErrors($validator)
                ->withInput();
        }

        // حساب المبلغ الإجمالي
        $totalAmount = $request->amount + $request->fee;
        
        // إنشاء رقم مرجعي فريد للحوالة
        $referenceNumber = Remittance::generateReferenceNumber();
        
        // إنشاء رمز التحويل للمستلم
        $transferCode = strtoupper(substr(md5(uniqid()), 0, 8));
        
        // إنشاء سجل الحوالة الجديدة
        $remittance = Remittance::create([
            'reference_number' => $referenceNumber,
            'sender_id' => $request->sender_id,
            'receiver_name' => $request->receiver_name,
            'receiver_phone' => $request->receiver_phone,
            'receiver_address' => $request->receiver_address,
            'receiver_id_number' => $request->receiver_id_number,
            'from_currency_id' => $request->from_currency_id,
            'to_currency_id' => $request->to_currency_id,
            'amount' => $request->amount,
            'fee' => $request->fee,
            'exchange_rate' => $request->exchange_rate,
            'total_amount' => $totalAmount,
            'status' => Remittance::STATUS_PENDING,
            'payment_method' => $request->payment_method,
            'destination_country' => $request->destination_country,
            'destination_city' => $request->destination_city,
            'transfer_purpose' => $request->transfer_purpose,
            'transfer_code' => $transferCode,
            'notes' => $request->notes,
            'expected_delivery_date' => $request->expected_delivery_date,
            'created_by' => Auth::id(),
            'updated_by' => Auth::id()
        ]);

        return redirect()->route('remittances.show', $remittance->id)
            ->with('success', 'تم إنشاء الحوالة بنجاح');
    }

    /**
     * عرض تفاصيل حوالة محددة
     */
    public function show(Remittance $remittance)
    {
        $remittance->load(['sender', 'fromCurrency', 'toCurrency', 'createdBy', 'updatedBy']);
        return view('remittances.show', compact('remittance'));
    }

    /**
     * عرض نموذج تعديل حوالة محددة
     */
    public function edit(Remittance $remittance)
    {
        // التحقق من إمكانية تعديل الحوالة
        if (in_array($remittance->status, [Remittance::STATUS_COMPLETED, Remittance::STATUS_CANCELLED, Remittance::STATUS_REFUNDED])) {
            return redirect()->route('remittances.show', $remittance->id)
                ->with('error', 'لا يمكن تعديل الحوالة في حالتها الحالية');
        }
        
        $customers = Customer::orderBy('name')->get();
        $currencies = Currency::active()->ordered()->get();
        $paymentMethods = [
            Remittance::PAYMENT_CASH => 'نقداً',
            Remittance::PAYMENT_BANK_TRANSFER => 'تحويل بنكي',
            Remittance::PAYMENT_CREDIT_CARD => 'بطاقة ائتمان',
            Remittance::PAYMENT_MOBILE_WALLET => 'محفظة إلكترونية'
        ];
        
        $statuses = [
            Remittance::STATUS_PENDING => 'معلقة',
            Remittance::STATUS_PROCESSING => 'قيد المعالجة',
            Remittance::STATUS_COMPLETED => 'مكتملة',
            Remittance::STATUS_CANCELLED => 'ملغاة',
            Remittance::STATUS_REFUNDED => 'مستردة',
            Remittance::STATUS_ON_HOLD => 'معلقة مؤقتاً'
        ];
        
        return view('remittances.edit', compact('remittance', 'customers', 'currencies', 'paymentMethods', 'statuses'));
    }

    /**
     * تحديث بيانات حوالة محددة في قاعدة البيانات
     */
    public function update(Request $request, Remittance $remittance)
    {
        // التحقق من إمكانية تعديل الحوالة
        if (in_array($remittance->status, [Remittance::STATUS_COMPLETED, Remittance::STATUS_CANCELLED, Remittance::STATUS_REFUNDED])) {
            return redirect()->route('remittances.show', $remittance->id)
                ->with('error', 'لا يمكن تعديل الحوالة في حالتها الحالية');
        }
        
        // التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'sender_id' => 'required|exists:customers,id',
            'receiver_name' => 'required|string|max:255',
            'receiver_phone' => 'required|string|max:20',
            'receiver_address' => 'required|string|max:500',
            'receiver_id_number' => 'required|string|max:50',
            'from_currency_id' => 'required|exists:currencies,id',
            'to_currency_id' => 'required|exists:currencies,id|different:from_currency_id',
            'amount' => 'required|numeric|min:0.01',
            'fee' => 'required|numeric|min:0',
            'exchange_rate' => 'required|numeric|min:0.000001',
            'payment_method' => 'required|string',
            'status' => 'required|string',
            'destination_country' => 'required|string|max:100',
            'destination_city' => 'required|string|max:100',
            'transfer_purpose' => 'required|string|max:255',
            'notes' => 'nullable|string|max:1000',
            'expected_delivery_date' => 'nullable|date',
            'actual_delivery_date' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return redirect()->route('remittances.edit', $remittance->id)
                ->withErrors($validator)
                ->withInput();
        }

        // حساب المبلغ الإجمالي
        $totalAmount = $request->amount + $request->fee;
        
        // تحديث بيانات الحوالة
        $remittance->sender_id = $request->sender_id;
        $remittance->receiver_name = $request->receiver_name;
        $remittance->receiver_phone = $request->receiver_phone;
        $remittance->receiver_address = $request->receiver_address;
        $remittance->receiver_id_number = $request->receiver_id_number;
        $remittance->from_currency_id = $request->from_currency_id;
        $remittance->to_currency_id = $request->to_currency_id;
        $remittance->amount = $request->amount;
        $remittance->fee = $request->fee;
        $remittance->exchange_rate = $request->exchange_rate;
        $remittance->total_amount = $totalAmount;
        $remittance->status = $request->status;
        $remittance->payment_method = $request->payment_method;
        $remittance->destination_country = $request->destination_country;
        $remittance->destination_city = $request->destination_city;
        $remittance->transfer_purpose = $request->transfer_purpose;
        $remittance->notes = $request->notes;
        $remittance->expected_delivery_date = $request->expected_delivery_date;
        
        // تحديث تاريخ التسليم الفعلي إذا تم تغيير الحالة إلى مكتملة
        if ($request->status == Remittance::STATUS_COMPLETED && !$remittance->actual_delivery_date) {
            $remittance->actual_delivery_date = now();
        } elseif ($request->status != Remittance::STATUS_COMPLETED) {
            $remittance->actual_delivery_date = $request->actual_delivery_date;
        }
        
        $remittance->updated_by = Auth::id();
        $remittance->save();

        return redirect()->route('remittances.show', $remittance->id)
            ->with('success', 'تم تحديث الحوالة بنجاح');
    }

    /**
     * حذف حوالة محددة من قاعدة البيانات
     */
    public function destroy(Remittance $remittance)
    {
        // التحقق من إمكانية حذف الحوالة
        if (!in_array($remittance->status, [Remittance::STATUS_PENDING, Remittance::STATUS_CANCELLED])) {
            return redirect()->route('remittances.show', $remittance->id)
                ->with('error', 'لا يمكن حذف الحوالة في حالتها الحالية');
        }
        
        $remittance->delete();

        return redirect()->route('remittances.index')
            ->with('success', 'تم حذف الحوالة بنجاح');
    }

    /**
     * تغيير حالة الحوالة
     */
    public function updateStatus(Request $request, Remittance $remittance)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|string',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return redirect()->route('remittances.show', $remittance->id)
                ->withErrors($validator)
                ->withInput();
        }
        
        // تحديث حالة الحوالة
        $oldStatus = $remittance->status;
        $remittance->status = $request->status;
        
        // إضافة الملاحظات إذا تم توفيرها
        if ($request->notes) {
            $remittance->notes = $remittance->notes . "\n" . now()->format('Y-m-d H:i') . " - تغيير الحالة من {$oldStatus} إلى {$request->status}: " . $request->notes;
        }
        
        // تحديث تاريخ التسليم الفعلي إذا تم تغيير الحالة إلى مكتملة
        if ($request->status == Remittance::STATUS_COMPLETED && !$remittance->actual_delivery_date) {
            $remittance->actual_delivery_date = now();
        }
        
        $remittance->updated_by = Auth::id();
        $remittance->save();

        return redirect()->route('remittances.show', $remittance->id)
            ->with('success', 'تم تحديث حالة الحوالة بنجاح');
    }

    /**
     * البحث عن الحوالات
     */
    public function search(Request $request)
    {
        $query = $request->input('query');
        
        $remittances = Remittance::where('reference_number', 'like', "%{$query}%")
            ->orWhere('receiver_name', 'like', "%{$query}%")
            ->orWhere('receiver_phone', 'like', "%{$query}%")
            ->orWhere('transfer_code', 'like', "%{$query}%")
            ->orWhereHas('sender', function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('phone', 'like', "%{$query}%");
            })
            ->with(['sender', 'fromCurrency', 'toCurrency'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return view('remittances.search', compact('remittances', 'query'));
    }

    /**
     * طباعة إيصال الحوالة
     */
    public function printReceipt(Remittance $remittance)
    {
        $remittance->load(['sender', 'fromCurrency', 'toCurrency', 'createdBy']);
        return view('remittances.receipt', compact('remittance'));
    }

    /**
     * التحقق من حالة الحوالة باستخدام الرقم المرجعي أو رمز التحويل
     */
    public function checkStatus(Request $request)
    {
        $reference = $request->input('reference');
        
        if (!$reference) {
            return view('remittances.check_status');
        }
        
        $remittance = Remittance::where('reference_number', $reference)
            ->orWhere('transfer_code', $reference)
            ->with(['fromCurrency', 'toCurrency'])
            ->first();
            
        if (!$remittance) {
            return view('remittances.check_status')
                ->with('error', 'لم يتم العثور على حوالة بهذا الرقم المرجعي أو رمز التحويل');
        }
        
        return view('remittances.check_status', compact('remittance'));
    }
}
