<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CurrencyController extends Controller
{
    /**
     * عرض قائمة العملات
     */
    public function index()
    {
        $currencies = Currency::ordered()->get();
        return view('currencies.index', compact('currencies'));
    }

    /**
     * عرض نموذج إضافة عملة جديدة
     */
    public function create()
    {
        return view('currencies.create');
    }

    /**
     * حفظ عملة جديدة في قاعدة البيانات
     */
    public function store(Request $request)
    {
        // التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:10|unique:currencies,code',
            'symbol' => 'required|string|max:10',
            'country' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'flag_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'decimal_places' => 'required|integer|min:0|max:6',
            'sort_order' => 'nullable|integer|min:0'
        ]);

        if ($validator->fails()) {
            return redirect()->route('currencies.create')
                ->withErrors($validator)
                ->withInput();
        }

        // معالجة ملف صورة العلم إذا تم تحميله
        $flagImagePath = null;
        if ($request->hasFile('flag_image')) {
            $flagImagePath = $request->file('flag_image')->store('flags', 'public');
        }

        // إنشاء سجل العملة الجديدة
        $currency = Currency::create([
            'name' => $request->name,
            'code' => strtoupper($request->code),
            'symbol' => $request->symbol,
            'country' => $request->country,
            'is_active' => $request->has('is_active'),
            'flag_image' => $flagImagePath,
            'decimal_places' => $request->decimal_places,
            'sort_order' => $request->sort_order ?? 999
        ]);

        return redirect()->route('currencies.index')
            ->with('success', 'تم إضافة العملة بنجاح');
    }

    /**
     * عرض تفاصيل عملة محددة
     */
    public function show(Currency $currency)
    {
        // الحصول على أسعار الصرف المرتبطة بهذه العملة
        $exchangeRates = $currency->fromExchangeRates()
            ->with('toCurrency')
            ->active()
            ->latest()
            ->get();

        return view('currencies.show', compact('currency', 'exchangeRates'));
    }

    /**
     * عرض نموذج تعديل عملة محددة
     */
    public function edit(Currency $currency)
    {
        return view('currencies.edit', compact('currency'));
    }

    /**
     * تحديث بيانات عملة محددة في قاعدة البيانات
     */
    public function update(Request $request, Currency $currency)
    {
        // التحقق من صحة البيانات المدخلة
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:10|unique:currencies,code,' . $currency->id,
            'symbol' => 'required|string|max:10',
            'country' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'flag_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'decimal_places' => 'required|integer|min:0|max:6',
            'sort_order' => 'nullable|integer|min:0'
        ]);

        if ($validator->fails()) {
            return redirect()->route('currencies.edit', $currency->id)
                ->withErrors($validator)
                ->withInput();
        }

        // معالجة ملف صورة العلم إذا تم تحميله
        if ($request->hasFile('flag_image')) {
            // حذف الصورة القديمة إذا كانت موجودة
            if ($currency->flag_image) {
                Storage::disk('public')->delete($currency->flag_image);
            }
            
            $flagImagePath = $request->file('flag_image')->store('flags', 'public');
            $currency->flag_image = $flagImagePath;
        }

        // تحديث بيانات العملة
        $currency->name = $request->name;
        $currency->code = strtoupper($request->code);
        $currency->symbol = $request->symbol;
        $currency->country = $request->country;
        $currency->is_active = $request->has('is_active');
        $currency->decimal_places = $request->decimal_places;
        $currency->sort_order = $request->sort_order ?? 999;
        $currency->save();

        return redirect()->route('currencies.index')
            ->with('success', 'تم تحديث العملة بنجاح');
    }

    /**
     * حذف عملة محددة من قاعدة البيانات
     */
    public function destroy(Currency $currency)
    {
        // التحقق من عدم وجود معاملات أو حوالات مرتبطة بالعملة
        $hasTransactions = $currency->fromTransactions()->count() > 0 || $currency->toTransactions()->count() > 0;
        $hasRemittances = $currency->fromRemittances()->count() > 0 || $currency->toRemittances()->count() > 0;
        
        if ($hasTransactions || $hasRemittances) {
            return redirect()->route('currencies.index')
                ->with('error', 'لا يمكن حذف هذه العملة لأنها مرتبطة بمعاملات أو حوالات');
        }

        // حذف صورة العلم إذا كانت موجودة
        if ($currency->flag_image) {
            Storage::disk('public')->delete($currency->flag_image);
        }

        // حذف أسعار الصرف المرتبطة بالعملة
        $currency->fromExchangeRates()->delete();
        $currency->toExchangeRates()->delete();

        // حذف العملة
        $currency->delete();

        return redirect()->route('currencies.index')
            ->with('success', 'تم حذف العملة بنجاح');
    }

    /**
     * تغيير حالة العملة (نشط/غير نشط)
     */
    public function toggleStatus(Currency $currency)
    {
        $currency->is_active = !$currency->is_active;
        $currency->save();

        return redirect()->route('currencies.index')
            ->with('success', 'تم تغيير حالة العملة بنجاح');
    }

    /**
     * البحث عن العملات
     */
    public function search(Request $request)
    {
        $query = $request->input('query');
        
        $currencies = Currency::where('name', 'like', "%{$query}%")
            ->orWhere('code', 'like', "%{$query}%")
            ->orWhere('country', 'like', "%{$query}%")
            ->ordered()
            ->get();

        return response()->json($currencies);
    }
}
